' TicTacToe.bas.
' Rev 1.0.0 William M Leue 7/9/2020

option default integer

const SIZE = 3
const SIZE2 = SIZE*SIZE
const EMPTY = 0
const BIGX = 1
const BIGO = 2
const BIGNUM = 100000
const true = 1
const false = 0
const RATIONAL = 0
const MTROWS = 41
const MTCOLS = 21
const MTINFO = 4

' Graphics params
const HCORR = 1.0  ' change to 0.75 for 16:9 monitor with no letterboxing
const CELLSIZE = 100
const LMARGIN = 50
const TMARGIN = 50
const XOSIZE = 75
const ZCOLOR = RGB(BLACK)
const BCOLOR = RGB(WHITE)
const XCOLOR = RGB(RED)
const OCOLOR = RGB(GREEN)
const WCOLOR = RGB(BLUE)
const LTHICK = 3
const XTHICK = 6
const OTHICK = 14
const NUMCMDS = 21
const STATUS_X = 20
const STATUS_Y = 550
const STATUS_LEN = 700
const STATUS_HEIGHT = 30

const CTLW = 144
const CTLH = 144
const CTLX = 550
const CTLY = 350
const CTLK = 36

' globals
dim board(SIZE, SIZE)
dim keys(NUMCMDS)
dim string which$

' computer move table
dim movetable(MTROWS, MTCOLS)
dim mlist(SIZE2)

active = 0
turn = 0
stkptr = 0
winner = 0
command = -1
key = 0
numk = 0
nzero = 0

' game counters
ungames = 0
tngames = 0
cngames = 0
p1games = 0
p2games = 0

' values for 1 or 2-player mode
nplayers = 0
p1name$ = ""
p2name$ = ""
player1 = 1
player2 = 2
player = 1
computer = 2
playersPiece = 0
computersPiece = 0
player1sPiece = 0
player2sPiece = 0

' currrentStartingPlayer determines who gets X.
' ponmove determines who has the next move.
' these will always be 1 or 2 after SetStartingPlayer is called.
' currentStartingPlayer alternates after each game, and
' ponmove alternates after each move.
currentStartingPlayer = 0
ponmove = 0

' Main program
'open "debug.txt" for output as #1
SetGraphics
MakeKeys
MakeMoveTables
ShowStartScreen
NewGame

do
  numk = keydown(0)
  if numk > 0 then
    key = keydown(1)
    if key <> 0 then
      if nplayers = 1 then
        DoCommand key
      else
        Do2PlayerCommand key
      end if  
      pause 250
    end if
  end if
loop
end

' Set up graphics mode
sub SetGraphics
  mode 1,8
end sub

' Make an array of supported key codes for user commands
Sub MakeKeys
  keys(0) = 43  : keys(1) = 45   : keys(2) = 132
  keys(3) = 135 : keys(4) = 129  : keys(5) = 137
  keys(6) = 130 : keys(7) = 53   : keys(8) = 131
  keys(9) = 134 : keys(10) = 128 : keys(11) = 136
  keys(12) = 55 : keys(13) = 56  : keys(14) = 57
  keys(15) = 52 : keys(16) = 53  : keys(17) = 54
  keys(18) = 49 : keys(19) = 50  : keys(20) = 51
end Sub

' Read the computer move tables from data
sub MakeMoveTables
  local row, col

  for row = 0 to MTROWS-1
    for col = 0 to MTCOLS
      read movetable(row, col)
    next col
  next row
end sub

' Respond to user key presses (1-player mode)
Sub DoCommand key
  local i, cx, m, np
  local code, type, index


  cx = -1
  for i = 0 to NUMCMDS-1
    if key = keys(i) then 
      cx = i
      exit for
    end if
  next i
  if cx < 0 then
    exit sub
  end if
  select case cx
    case 0
      NewGame
    case 1
      QuitGame
    case 2
       ComputerGoesFirst
    case else
      ShowMover
      m = MapKey(cx)
      if active = 1 and isLegalMove(m) then
        if ponmove = 1 then
          DoMove m, playersPiece
          getWinner code, type, index
          DrawBoard  code, type, index
          DeclareOutcome code
          if code <> 0 then 
            active = 0
          else
            ponmove = computer
            ShowMover
          end if
          if active = 1 then
            m = getComputerMove()
            pause 1000
            DoMove m, computersPiece
            getWinner code, type, index
            DrawBoard code, type, index
            DeclareOutcome code
            if code <> 0 then
              active = 0
            else
              ponmove = player
              ShowMover
            end if
          end if
        end if       
      else
        DrawStatus "Illegal Move"
      end if
  end select

end sub

' Respond to user key presses (2-player mode)
Sub Do2PlayerCommand key
  local i, cx, m, np , piece
  local code, type, index
  cx = -1
  if ponmove = player1 then
    piece = player1sPiece
  else
    piece = player2sPiece
  end if

  for i = 0 to NUMCMDS-1
    if key = keys(i) then 
      cx = i
      exit for
    end if
  next i
  if cx < 0 then
    exit sub
  end if
  select case cx
    case 0
      NewGame
    case 1
      QuitGame
    case 2
       ' nothing
    case else
      if active = 1 then
        ShowMover
        m = MapKey(cx)
        if active = 1 and isLegalMove(m) then
          DoMove m, piece
          getWinner code, type, index
          DrawBoard code, type, index
          DeclareOutcome code
          if code <> 0 then
            active = 0
          else
            ' flip players on the move during game
            if ponmove = player1 then
              ponmove = player2
              if player1 = currentStartingPlayer then
                piece = BIGO
              else
                piece = BIGX
              end if
            else
              ponmove = player1
              if player1 = currentStartingPlayer then
                piece = BIGX
              else
                player = BIGO
              end if
            end if
            ShowMover
          end if
        end if
      else
        DrawStatus "Illegal Move"
      end if
  end select

end sub

' Map keypad numeric presses into TicTacToe moves
' Dual entries for each move mean that they work
' whether the Num Lock is up or down.
function mapKey(cx)
  local m
  select case cx
    case 9  : m = 7
    case 10 : m = 8
    case 11 : m = 9
    case 6  : m = 4
    case 7  : m = 5
    case 8  : m = 6
    case 3  : m = 1
    case 4  : m = 2
    case 5  : m = 3
    case 12 : m = 7
    case 13 : m = 8
    case 14 : m = 9
    case 15 : m = 4
    case 16 : m = 5
    case 17 : m = 6
    case 18 : m = 1
    case 19 : m = 2
    case 20 : m = 3

  end select
  mapKey = m
end function 

' Report the outcome of a game
sub DeclareOutcome code
  if nplayers = 1 then  
    if code = BIGX then
      if currentStartingPlayer = player then
        DrawStatus "You Win, Game over"
        ungames = ungames+1
        active = 0
      else if currentStartingPlayer = computer then
        DrawStatus "Computer Wins, Game over"
        cngames = cngames+1
        active = 0
      end if
    else if code = BIGO then
      if currentStartingPlayer = player then
        DrawStatus "Computer Wins, Game over"
        cngames = cngames+1
      else
        DrawStatus "You Win, Game over"
        ungames = ungames+1
      end if
    else if code = 3 then
      DrawStatus "Cat wins, Game over"
      tngames = tngames+1
      active = 0
    else if winner = 0 then
      'DrawStatus "Your Move"
    end if
  else
    if code = BIGX then
      if currentStartingPlayer = player1 then
        DrawStatus p1name$ + " Wins, Game over"
        p1games = p1games+1
      else
        DrawStatus p2name$ + " Wins, Game over"
        p2games = p2games+1
      end if
    else if code = BIGO then
      if currentStartingPlayer = player1 then
        DrawStatus p2name$ + " Wins, Game over"
        p2games = p2games+1
      else
        DrawStatus p1name$ + " Wins, Game over"
        p1games = p1games+1
      end if
    else if code = 3 then
      DrawStatus "Cat wins, Game over"
      tngames = tngames+1
    else if winner = 0 then
      ShowMover
    end if
  end if  
  ShowScores
  if active = 0 then
    if ponmove = 1 then
      ponmove = 2
    else
      ponmove = 1
    end if
  end if  
end sub

' Set the player who goes first (X) for a new game
' so that they alternate taking X and O.
' the '0' case only happens when the program is first started.
sub SetStartingPlayer
  if nplayers = 1 then
    if currentStartingPlayer = 0 then
      currentStartingPlayer = player
      ponmove = player  
      playersPiece = BIGX
      computersPiece = BIGO      
    else if currentStartingPlayer = player then
      currentStartingPlayer = computer
      ponmove = computer
      computersPiece = BIGX
      playersPiece = BIGO
    else
      currentStartingPlayer = player
      ponmove = player
      playersPiece = BIGX
      computersPiece = BIGO      
    end if
  else
    if currentStartingPlayer = 0 then
      currentStartingPlayer = player1
      ponmove = 1
      player1sPiece = BIGX
      player2sPiece = BIGO
    else if currentStartingPlayer = player1 then
      currentStartingPlayer = player2
      ponmove = 2
      player1sPiece = BIGO
      player2sPiece = BIGX
    else
      currentStartingPlayer = player1
      ponmove = 1
      player1sPiece = BIGX
      player2sPiece = BIGO
    end if
  end if
  player = BIGX
end sub

' Print a status message below the board
sub DrawStatus message$
  box STATUS_X, STATUS_Y, STATUS_LENGTH, STATUS_HEIGHT,, ZCOLOR
  text STATUS_X+15, STATUS_Y+10, "                      "
  text STATUS_X+15, STATUS_Y+10, message$
end sub

' clean ending
sub QuitGame
  ' close #1
  end
end sub

' Initialize board and players for a new game
Sub NewGame()
  Local row, col, i

  For row = 0 To SIZE-1
    For col = 0 To SIZE-1
      board(row, col) = EMPTY
    Next col
  Next row
  for i = 0 to SIZE2-1
    mlist(i) = 0
  next i
  active = 1
  cls
  SetStartingPlayer
  DrawBoard 0
  ShowScores
  ShowControls
  ShowMover

  ' do initial computer move when computer has X.
  ' this makes the play loop simpler.
  if nplayers = 1 and ponmove = 2 then
    m = getComputerMove()
    pause 1000
    DoMove m, computersPiece
    getWinner code, type, index
    DrawBoard code, type, index
    DeclareOutcome code
    if code <> 0 then
      active = 0
    else
      ponmove = 1
      ShowMover
    end if
  end if  
End Sub

' Prompt the player on the move
sub ShowMover
  if nplayers = 1 then
    if ponmove = 1 then
        DrawStatus "Your Move"
      else
        DrawStatus "Computer's Move"
    end if
  else
    if ponmove = 1 then
      DrawStatus p1name$ + "'s Move"
    else
      DrawStatus p2name$ + "'s Move"
    end if
  end if
end sub

' Return the number of Xs and Os played
Function getNumPieces() As integer
  Local row, col, num
  num = 0
  For row = 0 To SIZE-1
    For col = 0 To SIZE-1
      If board(row, col) <> EMPTY Then
        num = num + 1
      End If
    Next col
  Next row
  getNumPieces = num
End Function

' Return 1 for a legal move, 0 for illegal
Function isLegalMove(move) As integer
  local legal, row, col
  if active = 0 then
    legal = 0
  else
    legal = 1
    row = SIZE-(move-1)\SIZE-1
    col = (move-1) mod SIZE
    If board(row, col) <> EMPTY Then legal = 0
  end if
  isLegalMove = legal
End Function

' Make the specified legal move for the specified player
sub DoMove move, who
  local row, col, winner, np

  row = SIZE-(move-1)\SIZE-1
  col = (move-1) mod SIZE
  board(row, col) = who
  np = GetNumPieces()
  if nplayers = 1 then
    mlist(np-1) = move
  end if
end sub

' Return the number of empty spaces, that is, legal moves
function getNumMoves() as integer
  local row, col, num

  num = 0
  for row = 0 to SIZE-1
    for col = 0 to SIZE-1
      if board(row, col) = EMPTY then
        num = num + 1
      end if
    next col
  next row
  getNumMoves = num
End function

' Get the computer move from lookup tables or 'rational'
' Plies 1 through 6 use move tables, and then 'rational'
' to the end of the game.
' Move tables are keyed first by the ply number, and then
' by previous opponent moves. It's simple and stupid.
function getComputerMove() as integer
  'local moves(SIZE*SIZE)
  local row, col, move, mrow, umove, key1, key2

  local ply = getNumPieces() + 1
  if ply > 1 then
    umove = mlist(ply-2)
  end if
  
  if ply > 6 then
    move = GetRationalMove()
    if move = 0 then
      move = GetRandomMove(computer)
    end if
    if move = 0 then
      ERROR "ply > 5: rational or random returned zero"
    end if
    getComputerMove = move
    exit function
  end if

  ' generate branch keys
  select case ply
    case 1
    case 2
    case 3
      key1 = 0
      key2 = 0
    case 4
      key1 = mlist(ply-4)
      key2 = 0
    case 5
      key1 = mlist(ply-4)
      key2 = 0
    case 6
      key1 = mlist(ply-4)
      key2 = mlist(ply-6)
  end select
 
  ' find the move table row that matches the current board:
  ' first, the ply number, and then 1 or 2 previous opponent
  ' moves.
  mrow = -1
  for row = 0 to MTROWS-1
    if movetable(row, 0) = ply then
      if key1 > 0  then
        if key2 > 0 then
          if movetable(row, 1) = key1 and movetable(row, 2) = key2 then
            mrow = row
            exit for
          end if
        else if  movetable(row, 1) = key1 then
          mrow = row
          exit for
        else
          continue for
        end if
      else if movetable(row, 0) = ply then 
        mrow = row
        exit for
      end if
    end if
  next row

  ' the relevant row of the move table has been located.
  ' now find the specific move to counter last user move
  if mrow >= 0 then
    nmoves = movetable(mrow, MTINFO-1)
    for col = MTINFO to MTINFO+2*(nmoves-1) step 2      
      if movetable(mrow, col) = umove then
        move = movetable(mrow, col+1)   
        getComputerMove = move
        exit function
      end if
    next col
  end if

  ' if lookup doesn't work, try a 'rational' or 'random' move
  if move = 0 then
    move = GetRationalMove()
    if move = 0 then
      move = GetRandomMove(computer)
    end if
  end if
  if move = 0 then
    ERROR "All move trials failed"
  end if
  getComputerMove = move
end function  

' If a winning move exists for the computer,
' then take it. If one exists for the player
' on his next move, block it.
function getRationalMove()
  local move, b, w

  w = 0
  w = getWinningMove(computersPiece)
  if w = 0 then
    w = getWinningMove(playersPiece)
  end if
  move = w
  getRationalMove = move
end function

' return the move code for a winning move by the
' specified player or zero if no such move exists
' 
function getWinningMove(who)
  local row, col, hit, n
  local cw, rw, foe
  
  foe = BIGO
  if who = BIGO then foe = BIGX  

  ' check for row winning move
  for row = 0 to SIZE-1
    hit = 1 : n = 0
    for col = 0 to SIZE-1
      if board(row, col) = foe then hit = 0
      if board(row, col) = who then n = n+1
      if board(row, col) = EMPTY then cw = col
    next col
    if hit = 1 and n = 2 then
      getWinningMove = (SIZE-row-1)*SIZE + cw + 1
      exit function
    end if
  next row

  ' check for column winning move
  for col = 0 to SIZE-1
    hit = 1 : n = 0
    for row = 0 to SIZE-1
      if board(row, col) = foe then hit = 0
      if board(row, col) = who then n = n+1
      if board(row, col) = EMPTY then rw = row
    next row
    if hit = 1 and n = 2 then
      getWinningMove = (SIZE-rw-1)*SIZE + col + 1 
      exit function
    end if
  next col
  
  ' check for diagonals winning moves
  hit = 1 : n = 0
  for row = 0 to SIZE-1
    col = row
    if board(row, col) = foe then hit = 0
    if board(row, col) = who then n = n+1
    if board(row, col) = EMPTY then rw = row
  next row    
  if hit = 1 and n = 2 then
    getWinningMove = (SIZE-rw-1)*SIZE + rw + 1
    exit function
  end if
  hit = 1 : n = 0
  for row = 0 to SIZE-1
    col = SIZE-row-1
    if board(row, col) = foe then hit = 0
    if board(row, col) = who then n = n+1
    if board(row, col) = EMPTY then 
      rw = row
      cw = col
    end if
  next row
  if hit = 1 and n = 2 then
    getWinningMove = (SIZE-rw-1)*SIZE + cw + 1
    exit function
  end if
  getWinningMove = 0
end function

' Return a  random legal move
function getRandomMove(who)
  local rlist(SIZE2)
  local i, p, n, cell

  n = 0
  for cell = 1 to 9
    if isLegalMove(cell) = 1 then
      rlist(n) = cell
  
      n = n+1
    end if
  next cell
  p = rnd()*n
  if p > n-1 then p = n-1
  getRandomMove = rlist(p)
end function

' Returns a code that says what the end game status is:
' code: 0 - Game in progress, no winner
'       1 - X Wins
'       2 - O Wins
'       3 - Cat Wins
' type: 1 - row win at row 'index'
'       2 - column win at column 'index'
'       3 - slant left diagonal win
'       4 - slant right diagonal win 
sub GetWinner code, type, index
  local cell, win, row, col    

  win = 0

  ' Check for a Row Win
  For row = 0 To SIZE-1
    win = 1
    cell = board(row, 0)
    code = cell
    If cell = EMPTY Then win = 0
    For col = 1 To SIZE-1
      ncell = board(row, col)
      If ncell <> cell Then win = 0
    Next col
    If win = 1 Then 
      type = 1
      index = row
      exit for
    end if
  Next row

  ' Check for a Column Win
  if win = 0 then
    For col = 0 To SIZE-1
      win = 1
      cell = board(0, col)
      If cell = EMPTY Then win = 0
      For row = 1 To SIZE-1
        If board(row, col) <> cell Then win = 0
      Next row
      If win = 1 Then
        type = 2
        index = col  
      exit for
      end if
    Next col
  end if
  
  ' Check for a Main Diagonal Win
  if win = 0 then
    win = 1
    cell = board(0, 0)
    If cell = EMPTY Then win = 0
    For row = 1 To SIZE-1
      If board(row, row) <> cell Then win = 0
    Next row
    if win = 1 then
      type = 3
      index = 0
      end if
    end if

  ' Check for a Minor Diagonal Win
  if win = 0 then
    win = 1
    cell = board(0, 2)
    if cell = EMPTY then win = 0
    For row = 1 To SIZE-1
      If board(row, SIZE - row - 1) <> cell Then win = 0
    Next row
    if win = 1 then
      type = 4
      index = 0 
    end if
  end if

  if win = 1 then
    code = cell
  else
    type = 0 : index = 0
    if GetNumPieces() = SIZE*SIZE then
      code = 3
    else 
      code = 0
    end if
  end if
End sub

' Draw the board with the current position using
' Mode 1 graphics on the CMM2.
' If there is a winner, the winning cells are hilited.
Sub DrawBoard wcode, type, index
  local x1, y1, x2, y2, row, col, cell
  local wloc

  x1 = HCORR*(LMARGIN + CELLSIZE)
  y1 = TMARGIN
  x2 = x1
  y2 = TMARGIN + 3*CELLSIZE
  line x1, y1, x2, y2, LTHICK, BCOLOR
  x1 = x1 + HCORR*CELLSIZE : x2 = x1
  line x1, y1, x2, y2, LTHICK, BCOLOR
  x1 = HCORR*LMARGIN
  y1 = TMARGIN + CELLSIZE
  x2 = x1 + HCORR*3*CELLSIZE
  y2 = y1
  line x1, y1, x2, y2, LTHICK, BCOLOR
  y1 = y1 + CELLSIZE
  y2 = y1
  line x1, y1, x2, y2, LTHICK, BCOLOR
  for row = 0 to SIZE-1
    for col = 0 to SIZE-1
      cell = board(row, col)
      select case cell
        case EMPTY
          ' do nothing
        case BIGX
          DrawBigX row, col
        case BIGO
          DrawBigO row, col
      end select
    next col
  next row
  if wcode <> 0 then
    select case type
      case 1
        x1 = HCORR*LMARGIN
        y1 = TMARGIN + index*CELLSIZE + CELLSIZE/2
        x2 = x1 + HCORR*3*CELLSIZE
        y2 = y1
        line x1, y1, x2, y2, 3, WCOLOR
      case 2
        x1 = HCORR*LMARGIN + HCORR*index*CELLSIZE + HCORR*CELLSIZE/2
        y1 = TMARGIN
        x2 = x1
        y2 = TMARGIN + 3*CELLSIZE
        line x1, y1, x2, y2, 3, WCOLOR
      case 3
        x1 = HCORR*LMARGIN
        y1 = TMARGIN
        x2 = HCORR*LMARGIN + HCORR*3*CELLSIZE
        y2 = TMARGIN + 3*CELLSIZE
        line x1, y1-1, x2, y2-1,, WCOLOR
        line x1, y1, x2, y2,, WCOLOR 
        line x1, y1+2, x2, y2+1,, WCOLOR
      case 4
        x1 = HCORR*LMARGIN + HCORR*3*CELLSIZE
        y1 = TMARGIN 
        x2 = HCORR*LMARGIN
        y2 = TMARGIN + 3*CELLSIZE
        line x1, y1-1, x2, y2-1,, WCOLOR
        line x1, y1, x2, y2,, WCOLOR
        line x1, y1+1, x2, y2+1,, WCOLOR
    end select
  end if
End Sub

' Draw a Big X in the specified cell
Sub DrawBigX row, col
  local cx, cy, s1, s2, s3, s4
  local xv(5), yv(5)
  cx = HCORR*LMARGIN + HCORR*col*CELLSIZE + HCORR*CELLSIZE/2
  cy = TMARGIN + row*CELLSIZE + CELLSIZE/2
  s1 = HCORR*XOSIZE/2
  s2 = HCORR*XTHICK/2
  s3 = XOSIZE/2
  s4 = XTHICK/2
  xv(0) = cx - s1 + s2
  yv(0) = cy - s3 - s4
  xv(1) = cx + s1 + s2
  yv(1) = cy + s3 - s4
  xv(2) = cx + s1 - s2
  yv(2) = cy + s3 + s4
  xv(3) = cx - s1 - s2
  yv(3) = cy - s3 + s4
  xv(4) = xv(0)
  yv(4) = yv(0)
  polygon 5, xv(), yv(), XCOLOR, XCOLOR
  xv(0) = cx - s1 - s2
  yv(0) = cy + s3 - s4
  xv(1) = cx + s1 - s2
  yv(1) = cy - s3 - s4
  xv(2) = cx + s1 + s2
  yv(2) = cy - s3 + s4
  xv(3) = cx - s1 + s2
  yv(3) = cy + s3 + s4
  xv(4) = xv(0)
  yv(4) = yv(0)
  polygon 5,  xv(), yv(), XCOLOR, XCOLOR 
end sub

' Draw a Big O in the specified cell
sub DrawBigO row, col
  local cx, cy, r1
  cx = HCORR*LMARGIN + HCORR*col*CELLSIZE + HCORR*CELLSIZE/2
  cy = TMARGIN + row*CELLSIZE + CELLSIZE/2
  r1 = (XOSIZE - OTHICK)/2
  circle cx, cy, XOSIZE/2,, HCORR, OCOLOR, OCOLOR
  circle cx, cy, r1,, HCORR, RGB(BLACK), RGB(BLACK)
end sub

sub ShowStartScreen
  cls
  text 400, 20, "Welcome to Classic Tic-Tac-Toe!", "CB"
  text 400, 40, "First to get three-in-a-row wins.", "CB"
  text 400, 60, "X goes first", "CB"

  do
    ok = 1
   text 10, 90, ""
    input "1 or 2 players? ", ans$
    if ans$ <> "1" and ans$ <> "2" then 
      ok = 0
      text 10, 90, "                    "
    end if
  loop until ok = 1
  nplayers = val(ans$)
  if nplayers = 2 then
    Show2PlayerScreen
    exit sub
  else
  end if
  
  text 400, 100, "You start as X, computer is O", "CB"
  text 400, 120, "and then you alternate sides.", "CB"
    
  text 30, 400, "Press any key to start..."
  do
  loop until INKEY$ <> ""
end sub

' Show the game scores
sub ShowScores
  if nplayers = 1 then  
    text 600, 60, "You",         CB,,, RGB(WHITE)
    text 660, 60, "Cat",         CB,,, RGB(WHITE)
    text 720, 60, "Computer",    CB,,, RGB(WHITE)
    text 600, 75, "---",         CB,,, RGB(WHITE)
    text 660, 75, "---",         CB,,, RGB(WHITE)
    text 720, 75, "--------",    CB,,, RGB(WHITE)
    text 600, 90, str$(ungames), CB,,, RGB(WHITE)
    text 660, 90, str$(tngames), CB,,, RGB(WHITE)
    text 720, 90, str$(cngames), CB,,, RGB(WHITE)
  else
    text 600, 60, p1name$,       CB,,, RGB(WHITE)
    text 660, 60, "Cat",         CB,,, RGB(WHITE)
    text 720, 60, p2name$,       CB,,, RGB(WHITE)
    text 600, 75, string$(len(p1name$), "-"), CB,,, RGB(WHITE)
    text 660, 75, "---",         CB,,, RGB(WHITE)
    text 720, 75, string$(len(p2name$), "-"), CB,,, RGB(WHITE)
    text 600, 90, str$(p1games), CB,,, RGB(WHITE)
    text 660, 90, str$(tngames), CB,,, RGB(WHITE)
    text 720, 90, str$(p2games), CB,,, RGB(WHITE)
  end if
end sub

' Show the game controls
sub ShowControls
  local row, col, cx, ry, tx, ty, c

  text CTLX+CTLW\2, CTLY-30, "Controls", "CB"
  for row = 0 to 4
    ry = CTLY+row*CTLK
    for col = 0 to 3
      cx = CTLX+col*CTLK
      c = RGB(WHITE)
      if row = 1 and col = 3 then
        rbox cx, ry, CTLK, 2*CTLK,, RGB(GREEN)
        text cx+CTLK\2, ry+CTLK, "+", "CB"
        text CTLX+CTLW+10, ry+CTLK, "New Game", "LT",,, RGB(GREEN)
      else if row = 2 and col = 3 then
        ' skip
      else if row = 3 and col = 3 then
        rbox cx, ry, CTLK, 2*CTLK,, c
        text cx+CTLK\2, ry+5, "Enter", "CTV"
      else if row = 4 and col = 3 then
        ' skip
      else if row = 4 and col = 0 then
        rbox cx, ry, 2*CTLK, CTLK,, c
        text cx+CTLK, ry+10, "0", "CT"
      else if row = 4 and col = 1 then
        ' skip
      else
        c = RGB(WHITE)
        if row = 0 and col = 3 then c = RGB(RED)
        if row >= 1 and row <= 3 and col <= 2 then c = RGB(BLUE)
        rbox cx, ry, CTLK, CTLK,, c
        tx = cx + 17
        ty = ry + 23
        select case row
          case 0
            select case col
              case 0 : text tx, ty, "NL", "CB"
              case 1 : text tx, ty, "/", "CB"
              case 2 : text tx, ty, "*", "CB"
              case 3 : text tx, ty, "-", "CB"
            end select
          case 1
            select case col 
              case 0 : text tx, ty, "7", "CB"
              case 1 : text tx, ty, "8", "CB"
              case 2 : text tx, ty, "9", "CB"
            end select
          case 2
            select case col 
              case 0 : text tx, ty, "4", "CB"
              case 1 : text tx, ty, "5", "CB"
              case 2 : text tx, ty, "6", "CB"
            end select
          case 3
            select case col 
              case 0 : text tx, ty, "1", "CB"
              case 1 : text tx, ty, "2", "CB"
              case 2 : text tx, ty, "3", "CB"
            end select
        end select
      end if
    next col
  next row
  text CTLX+CTLW+10, CTLY, "QUIT", "LT",,, RGB(RED)
  text CTLX-10, CTLY+CTLH\2, "Moves", "RT",,, RGB(BLUE)

end sub

' Setup screen for 2-player mode
sub Show2PlayerScreen
  cls
  text 10, 20, ""
  input "What is player 1's name?: ", p1name$
  text 10, 35, ""
  input "What is player 2's name?: ", p2name$

  text 10, 50, "Press any key to start..."
  do
    x$ = INKEY$
  loop until x$ <> ""
  player = BIGX
  ponmove = 1
end sub

'Computer Move tables
' Table Structure:
' Cell 0: ply number from 1
' Cell 1,2: branch keys1, 2: key2 only used for ply 6
' Cell 3: number of moves for branch
' Cell 4..3+2N: moves
' Computer moves first: odd plies
' User moves first: even plies
' The MTROWS constant MUST match the number of
' rows in this table!
data 1,0,0,1,0,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
data 2,0,0,9,7,5,8,5,9,5,4,5,5,7,6,5,1,5,2,5,3,5
data 3,0,0,8,8,5,9,5,4,5,5,3,6,5,1,5,2,5,3,1,0,0
data 4,7,0,4,6,3,2,4,3,2,4,1,0,0,0,0,0,0,0,0,0,0
data 4,8,0,5,4,7,6,9,1,7,2,4,3,6,0,0,0,0,0,0,0,0
data 4,9,0,7,7,8,8,7,4,7,6,3,1,4,2,6,3,6,0,0,0,0
data 4,4,0,5,8,7,9,8,6,7,2,1,3,2,0,0,0,0,0,0,0,0
data 4,5,0,7,3,4,4,6,1,9,2,8,9,1,8,2,6,4,0,0,0,0
data 4,6,0,6,7,8,8,9,4,8,1,2,2,3,3,9,0,0,0,0,0,0
data 4,2,0,5,7,4,8,4,9,6,4,1,6,3,0,0,0,0,0,0,0,0
data 4,1,0,3,4,7,4,7,2,3,0,0,0,0,0,0,0,0,0,0,0,0
data 4,3,0,3,7,4,8,6,4,1,0,0,0,0,0,0,0,0,0,0,0,0
data 5,8,0,1,3,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
data 5,9,0,1,3,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
data 5,4,0,1,3,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
data 5,5,0,7,8,2,9,1,4,6,6,4,1,9,2,8,3,1,0,0,0,0
data 5,6,0,1,3,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
data 5,1,0,1,3,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
data 5,2,0,1,3,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
data 5,3,0,1,4,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
data 6,6,7,2,2,1,3,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0
data 6,2,7,1,6,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
data 6,6,8,1,1,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
data 6,8,1,1,3,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
data 6,2,8,1,6,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
data 6,4,3,1,4,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
data 6,4,9,1,2,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
data 6,2,9,1,4,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
data 6,8,4,1,3,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
data 6,9,4,1,2,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
data 6,6,4,1,3,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
data 6,2,4,1,9,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
data 6,3,4,1,8,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
data 6,5,3,1,1,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
data 6,7,6,1,2,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
data 6,8,6,1,1,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
data 6,4,6,1,2,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
data 6,1,6,1,8,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
data 6,2,6,1,7,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
data 6,7,3,1,6,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
data 6,8,3,1,4,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0

' current number of rows: 41
